<?php
declare(strict_types=1);

final class Security
{
    public static function applySecurityHeaders(): void
    {
        header('X-Frame-Options: SAMEORIGIN');
        header('X-Content-Type-Options: nosniff');
        header('Referrer-Policy: strict-origin-when-cross-origin');
        header('Content-Security-Policy: default-src \'self\'; img-src \'self\' data:; script-src \'self\'; style-src \'self\' \'unsafe-inline\';');
    }

    public static function enforceSessionFreshness(): void
    {
        if (!isset($_SESSION['session_fresh'])) {
            $_SESSION['session_fresh'] = time();
            return;
        }

        $timeout = (int) config('SESSION_TIMEOUT', 3600);
        if (time() - (int) $_SESSION['session_fresh'] > $timeout) {
            session_regenerate_id(true);
            $_SESSION['session_fresh'] = time();
        }
    }

    public static function isMaintenanceMode(): bool
    {
        try {
            $settings = DB::query('SELECT maintenance_mode FROM settings WHERE id = 1')->fetch();
            return (bool) ($settings['maintenance_mode'] ?? false);
        } catch (Throwable $e) {
            return false;
        }
    }

    public static function rateLimit(string $key, int $limit, int $seconds): bool
    {
        $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        $identifier = hash('sha256', $key . '|' . $ip);
        $stmt = DB::query('SELECT * FROM rate_limits WHERE identifier = :id LIMIT 1', ['id' => $identifier]);
        $row = $stmt->fetch();
        $now = time();

        if (!$row) {
            DB::query('INSERT INTO rate_limits (identifier, attempts, expires_at) VALUES (:id, 1, :expires)', [
                'id' => $identifier,
                'expires' => date('Y-m-d H:i:s', $now + $seconds),
            ]);
            return true;
        }

        if (strtotime($row['expires_at']) < $now) {
            DB::query('UPDATE rate_limits SET attempts = 1, expires_at = :expires WHERE identifier = :id', [
                'id' => $identifier,
                'expires' => date('Y-m-d H:i:s', $now + $seconds),
            ]);
            return true;
        }

        if ((int) $row['attempts'] >= $limit) {
            return false;
        }

        DB::query('UPDATE rate_limits SET attempts = attempts + 1 WHERE identifier = :id', [
            'id' => $identifier,
        ]);
        return true;
    }
}
