<?php
declare(strict_types=1);

final class LicenseService
{
    public static function verify(string $licenseKey, string $pluginSlug, array $payload): array
    {
        $stmt = DB::query(
            'SELECT licenses.*, plugins.slug AS plugin_slug FROM licenses JOIN plugins ON plugins.id = licenses.plugin_id WHERE licenses.license_key = :key LIMIT 1',
            ['key' => $licenseKey]
        );
        $license = $stmt->fetch();

        if (!$license || $license['plugin_slug'] !== $pluginSlug) {
            return ['ok' => false, 'code' => 'LICENSE_NOT_FOUND', 'message' => 'License not found.'];
        }

        $expiresAt = new DateTimeImmutable($license['expires_at']);
        $graceDays = (int) (config('LICENSE_GRACE_DAYS', 3));
        $graceUntil = $expiresAt->modify('+' . $graceDays . ' days');
        $now = new DateTimeImmutable('now');

        if ($license['status'] !== 'active' && $now > $graceUntil) {
            return ['ok' => false, 'code' => 'LICENSE_EXPIRED', 'message' => 'License expired.'];
        }

        $serverId = $payload['server_id'] ?? '';
        $signature = LicenseSigner::sign(
            $licenseKey,
            $pluginSlug,
            $license['expires_at'],
            $serverId,
            config('APP_KEY')
        );

        return [
            'ok' => true,
            'code' => 'LICENSE_VALID',
            'message' => 'License is valid.',
            'data' => [
                'user_id' => (int) $license['user_id'],
                'plugin_id' => (int) $license['plugin_id'],
                'expires_at' => $license['expires_at'],
                'grace' => $now > $expiresAt,
                'signature' => $signature,
            ],
        ];
    }
}
