<?php
declare(strict_types=1);

final class LicenseApiController
{
    public static function verify(): void
    {
        $payload = json_decode(file_get_contents('php://input'), true) ?: [];
        $token = $_SERVER['HTTP_X_API_TOKEN'] ?? '';

        if (!Security::rateLimit('api_verify_' . $token, 60, 60)) {
            self::respond(['ok' => false, 'code' => 'RATE_LIMIT', 'message' => 'Rate limit exceeded.'], 429);
            return;
        }

        $tokenRow = DB::query('SELECT * FROM api_tokens WHERE token = :token AND is_active = 1', [
            'token' => $token,
        ])->fetch();

        if (!$tokenRow) {
            self::respond(['ok' => false, 'code' => 'INVALID_TOKEN', 'message' => 'Invalid API token.'], 401);
            return;
        }

        $licenseKey = (string) ($payload['license_key'] ?? '');
        $pluginSlug = (string) ($payload['plugin_slug'] ?? '');

        if ($licenseKey === '' || $pluginSlug === '') {
            self::respond(['ok' => false, 'code' => 'INVALID_REQUEST', 'message' => 'Missing license key or plugin slug.'], 422);
            return;
        }

        $result = LicenseService::verify($licenseKey, $pluginSlug, $payload);
        self::logApiCall($token, $payload, $result);
        self::respond($result, $result['ok'] ? 200 : 403);
    }

    public static function latest(): void
    {
        $slug = $_GET['slug'] ?? '';
        if ($slug === '') {
            self::respond(['ok' => false, 'code' => 'INVALID_REQUEST', 'message' => 'Missing slug.'], 422);
            return;
        }

        $stmt = DB::query('SELECT id FROM plugins WHERE slug = :slug AND is_active = 1', ['slug' => $slug]);
        $plugin = $stmt->fetch();
        if (!$plugin) {
            self::respond(['ok' => false, 'code' => 'NOT_FOUND', 'message' => 'Plugin not found.'], 404);
            return;
        }

        $version = DB::query('SELECT * FROM plugin_versions WHERE plugin_id = :id AND is_public = 1 ORDER BY created_at DESC LIMIT 1', [
            'id' => $plugin['id'],
        ])->fetch();

        if (!$version) {
            self::respond(['ok' => false, 'code' => 'NO_VERSION', 'message' => 'No public version.'], 404);
            return;
        }

        $token = bin2hex(random_bytes(24));
        DB::query('INSERT INTO download_tokens (token, version_id, expires_at) VALUES (:token, :version, :expires)', [
            'token' => $token,
            'version' => $version['id'],
            'expires' => date('Y-m-d H:i:s', time() + 600),
        ]);

        self::respond([
            'ok' => true,
            'code' => 'LATEST_VERSION',
            'message' => 'Latest version ready.',
            'data' => [
                'version' => $version['version'],
                'changelog' => $version['changelog_md'],
                'download_url' => url('download.php?token=' . $token),
            ],
        ]);
    }

    public static function status(): void
    {
        $maintenance = Security::isMaintenanceMode();
        self::respond([
            'ok' => !$maintenance,
            'code' => $maintenance ? 'MAINTENANCE' : 'OK',
            'message' => $maintenance ? 'Maintenance mode.' : 'Service is healthy.',
            'data' => ['maintenance' => $maintenance],
        ]);
    }

    private static function respond(array $payload, int $status = 200): void
    {
        http_response_code($status);
        header('Content-Type: application/json');
        echo json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    }

    private static function logApiCall(string $token, array $request, array $response): void
    {
        DB::query('INSERT INTO api_logs (provider_token, request_json, response_json, ip_address, created_at) VALUES (:token, :req, :res, :ip, :created)', [
            'token' => $token,
            'req' => json_encode($request),
            'res' => json_encode($response),
            'ip' => $_SERVER['REMOTE_ADDR'] ?? null,
            'created' => now(),
        ]);
    }
}
